<?php
declare(strict_types=1);

use DateTime;

require_once __DIR__ . '/../includes/app.php';
require_once __DIR__ . '/../includes/mailer.php';

ensure_session();

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Location: index.php');
    exit;
}

$lang = resolve_language($_POST['lang'] ?? null);
$formData = [];

foreach ($_POST as $key => $value) {
    if (is_string($value)) {
        $formData[$key] = trim($value);
    }
}

unset($formData['_token'], $formData['lang']);

if (!valid_csrf($_POST['_token'] ?? null)) {
    set_flash('error', translate('errors.csrf', $lang), [], $formData, ['lang' => $lang]);
    header('Location: index.php?lang=' . urlencode($lang));
    exit;
}

$rateKey = rate_limit_key('public_form');
if (hit_rate_limit($rateKey, 5, 60)) {
    set_flash('error', translate('errors.rate_limited', $lang), [], $formData, ['lang' => $lang]);
    header('Location: index.php?lang=' . urlencode($lang));
    exit;
}

[$errors, $payload] = validate_application($formData, $lang);

if ($errors) {
    set_flash('error', translate('errors.summary', $lang), $errors, $formData, ['lang' => $lang]);
    header('Location: index.php?lang=' . urlencode($lang));
    exit;
}

$columns = array_keys($payload);

try {
    $placeholders = ':' . implode(', :', $columns);
    $statement = db()->prepare(
        sprintf(
            'INSERT INTO welms_applications (%s) VALUES (%s)',
            implode(', ', $columns),
            $placeholders
        )
    );

    foreach ($payload as $column => $value) {
        if ($value === null) {
            $statement->bindValue(':' . $column, null, PDO::PARAM_NULL);
        } elseif ($column === 'nb_employes') {
            $statement->bindValue(':' . $column, $value, PDO::PARAM_INT);
        } elseif ($column === 'date_signature') {
            $statement->bindValue(':' . $column, $value);
        } else {
            $statement->bindValue(':' . $column, $value, PDO::PARAM_STR);
        }
    }

    $statement->execute();
    reset_rate_limit($rateKey);

    if (!send_submission_notification($payload)) {
        error_log('Notification e-mail non envoyée.');
    }

    set_flash('success', translate('messages.success', $lang), [], [], ['lang' => $lang]);
} catch (Throwable $exception) {
    error_log($exception->getMessage());
    set_flash('error', translate('errors.server', $lang), [], $formData, ['lang' => $lang]);
}

header('Location: index.php?lang=' . urlencode($lang));
exit;

function sanitize_value(?string $value): ?string
{
    if ($value === null) {
        return null;
    }

    $trimmed = trim($value);

    return $trimmed === '' ? null : $trimmed;
}

function validate_application(array $data, string $lang): array
{
    $errors = [];
    $payload = [];

    $required = ['nom_entreprise', 'courriel_entreprise', 'resp_cf', 'date_signature', 'nom_signataire'];
    $maxLengths = [
        'nom_entreprise' => 255,
        'ville' => 120,
        'province' => 120,
        'code_postal' => 30,
        'courriel_entreprise' => 190,
        'type_entreprise' => 190,
        'etablie_depuis' => 50,
        'mode_paiement' => 190,
        'livrera' => 255,
        'ville_2' => 120,
        'province_2' => 120,
        'code_postal_2' => 30,
        'tel_2' => 60,
        'courriel_2' => 190,
        'resp_cf' => 255,
        'resp_achats' => 255,
        'resp_tech' => 255,
        'president' => 255,
        'type_industrie' => 190,
        'ref1_nom' => 255,
        'ref1_personne' => 255,
        'ref1_tel' => 60,
        'ref2_nom' => 255,
        'ref2_personne' => 255,
        'ref2_tel' => 60,
        'ref3' => 255,
        'ref4' => 255,
        'ref1_2' => 255,
        'ref2_2' => 255,
        'ref1_3' => 255,
        'ref2_3' => 255,
        'nom_entreprise_aut' => 255,
        'nom_signataire' => 190,
        'titre_signataire' => 190,
        'adresse_fact' => 255,
        'adresse_diff' => 255,
        'tel1' => 60,
        'fax1' => 60,
        'text3' => 2000,
    ];

    $emailFields = ['courriel_entreprise', 'courriel_2'];
    $phoneFields = ['tel_2', 'ref1_tel', 'ref2_tel', 'tel1', 'fax1'];

    foreach ($required as $field) {
        if (empty($data[$field])) {
            $errors[] = translate('errors.required', $lang, ['field' => field_label($field, $lang)]);
        }
    }

    foreach ($maxLengths as $field => $max) {
        if (!empty($data[$field]) && mb_strlen($data[$field]) > $max) {
            $errors[] = translate('errors.length', $lang, [
                'field' => field_label($field, $lang),
                'max' => (string) $max,
            ]);
        }
    }

    foreach ($emailFields as $field) {
        if (!empty($data[$field]) && !filter_var($data[$field], FILTER_VALIDATE_EMAIL)) {
            $errors[] = translate('errors.email', $lang, ['field' => field_label($field, $lang)]);
        }
    }

    foreach ($phoneFields as $field) {
        if (!empty($data[$field]) && !preg_match('/^[0-9+().\-\s]{6,}$/', $data[$field])) {
            $errors[] = translate('errors.phone', $lang, ['field' => field_label($field, $lang)]);
        }
    }

    if (!empty($data['nb_employes']) && (!ctype_digit((string) $data['nb_employes']) || (int) $data['nb_employes'] < 0)) {
        $errors[] = translate('errors.integer', $lang, ['field' => field_label('nb_employes', $lang)]);
    }

    if (!empty($data['date_signature'])) {
        $date = DateTime::createFromFormat('Y-m-d', $data['date_signature']);
        if (!$date || $date->format('Y-m-d') !== $data['date_signature']) {
            $errors[] = translate('errors.date', $lang, ['field' => field_label('date_signature', $lang)]);
        }
    }

    if (!empty($data['exotaxes']) && !in_array($data['exotaxes'], ['oui', 'non'], true)) {
        $errors[] = translate('errors.invalid_choice', $lang, ['field' => field_label('exotaxes', $lang)]);
    }

    $columns = [
        'nom_entreprise', 'ville', 'province', 'code_postal', 'courriel_entreprise', 'type_entreprise', 'etablie_depuis',
        'mode_paiement', 'livrera', 'ville_2', 'province_2', 'code_postal_2', 'tel_2', 'courriel_2', 'resp_cf',
        'resp_achats', 'resp_tech', 'president', 'nb_employes', 'type_industrie', 'ref1_nom', 'ref1_personne', 'ref1_tel',
        'ref2_nom', 'ref2_personne', 'ref2_tel', 'ref3', 'ref4', 'ref1_2', 'ref2_2', 'ref1_3', 'ref2_3', 'date_signature',
        'nom_entreprise_aut', 'nom_signataire', 'titre_signataire', 'adresse_fact', 'adresse_diff', 'tel1', 'fax1',
        'exotaxes', 'text3'
    ];

    foreach ($columns as $column) {
        if ($column === 'nb_employes') {
            $payload[$column] = isset($data[$column]) && $data[$column] !== '' ? (int) $data[$column] : null;
        } elseif ($column === 'date_signature') {
            $payload[$column] = sanitize_value($data[$column] ?? null);
        } else {
            $payload[$column] = sanitize_value($data[$column] ?? null);
        }
    }

    return [$errors, $payload];
}
