<?php

declare(strict_types=1);

require_once __DIR__ . '/app.php';

function mail_config(): array
{
    return app_config()['mail'] ?? [];
}

function smtp_enabled(): bool
{
    $config = mail_config();
    return !empty($config['host']) && !empty($config['to_email']);
}

function send_submission_notification(array $payload): bool
{
    if (!smtp_enabled()) {
        return false;
    }

    $config = mail_config();
    $company = $payload['nom_entreprise'] ?? 'Entreprise inconnue';
    $subject = 'Nouvelle demande WELMS / New WELMS request - ' . $company;

    $lines = [
        'Nouvelle demande d’ouverture de compte WELMS.',
        'New WELMS account-opening request.',
        '',
        'Entreprise / Company: ' . ($payload['nom_entreprise'] ?? 'N/D'),
        'Responsable CF: ' . ($payload['resp_cf'] ?? 'N/D'),
        'Courriel / Email: ' . ($payload['courriel_entreprise'] ?? 'N/D'),
        'Ville / City: ' . ($payload['ville'] ?? 'N/D'),
        'Téléphone / Phone: ' . ($payload['tel1'] ?? 'N/D'),
        'Date de signature / Signature date: ' . ($payload['date_signature'] ?? 'N/D'),
        'Mode de paiement / Payment method: ' . ($payload['mode_paiement'] ?? 'N/D'),
        '',
        'Voir le tableau de bord pour tous les détails. / See the dashboard for full details.',
    ];

    $body = implode("\n", $lines);

    $recipients = array_filter(array_map('trim', explode(',', (string) $config['to_email'])));
    $overallSuccess = false;

    foreach ($recipients as $recipient) {
        if (smtp_send_mail(
            $recipient,
            $config['to_name'] ?: $recipient,
            $subject,
            $body
        )) {
            $overallSuccess = true;
        }
    }

    return $overallSuccess;
}

function smtp_send_mail(string $toEmail, string $toName, string $subject, string $body): bool
{
    $config = mail_config();
    if (empty($config['host'])) {
        return false;
    }

    $encryption = strtolower($config['encryption'] ?? 'tls');
    $port = $config['port'] ?: ($encryption === 'ssl' ? 465 : 587);
    $transport = $encryption === 'ssl' ? 'ssl://' : 'tcp://';

    $contextOptions = [];
    if ($encryption !== 'none') {
        $contextOptions['ssl'] = [
            'verify_peer' => true,
            'verify_peer_name' => true,
            'allow_self_signed' => false,
        ];

        if (!empty($config['ca_file'])) {
            $contextOptions['ssl']['cafile'] = $config['ca_file'];
        }
    }

    $socket = @stream_socket_client(
        $transport . $config['host'] . ':' . $port,
        $errno,
        $errstr,
        30,
        STREAM_CLIENT_CONNECT,
        stream_context_create($contextOptions)
    );

    if (!$socket) {
        error_log('SMTP connection failed: ' . $errstr);
        return false;
    }

    stream_set_timeout($socket, 30);

    $readResponse = function () use ($socket) {
        $response = '';
        while ($line = fgets($socket, 515)) {
            $response .= $line;
            if (isset($line[3]) && $line[3] === ' ') {
                break;
            }
        }
        return $response;
    };

    $sendCommand = function (string $command) use ($socket, $readResponse) {
        fwrite($socket, $command . "\r\n");
        return $readResponse();
    };

    $expect = function (string $response, array $codes, string $context): void {
        $code = (int) substr($response, 0, 3);
        if (!in_array($code, $codes, true)) {
            throw new RuntimeException(sprintf('%s (code %s)', $context, trim($response)));
        }
    };

    try {
        $expect($readResponse(), [220], 'Connexion SMTP refusée');
        $expect($sendCommand('EHLO welms.local'), [250], 'EHLO initial');

        if ($encryption === 'tls') {
            $expect($sendCommand('STARTTLS'), [220], 'STARTTLS refusé');
            if (!stream_socket_enable_crypto($socket, true, STREAM_CRYPTO_METHOD_TLS_CLIENT)) {
                throw new RuntimeException('Activation TLS impossible');
            }
            $expect($sendCommand('EHLO welms.local'), [250], 'EHLO après TLS');
        }

        if (!empty($config['username'])) {
            $expect($sendCommand('AUTH LOGIN'), [334], 'AUTH LOGIN refusé');
            $expect($sendCommand(base64_encode($config['username'])), [334], 'Utilisateur SMTP invalide');
            $expect($sendCommand(base64_encode($config['password'] ?? '')), [235], 'Mot de passe SMTP invalide');
        }

        $fromEmail = $config['from_email'] ?: $config['username'];
        $fromName = $config['from_name'] ?: 'WELMS';

        $expect($sendCommand('MAIL FROM: <' . $fromEmail . '>'), [250], 'MAIL FROM refusé');
        $expect($sendCommand('RCPT TO: <' . $toEmail . '>'), [250, 251], 'RCPT TO refusé');
        $expect($sendCommand('DATA'), [354], 'DATA refusé');

        $normalizedBody = preg_replace("/\r?\n/", "\r\n", $body);
        $normalizedBody = str_replace("\r\n.", "\r\n..", $normalizedBody ?? '');

        $headers = [
            'From: ' . encode_address($fromName, $fromEmail),
            'To: ' . encode_address($toName, $toEmail),
            'Subject: ' . encode_header($subject),
            'MIME-Version: 1.0',
            'Content-Type: text/plain; charset=UTF-8',
            'Content-Transfer-Encoding: 8bit',
        ];

        $payload = implode("\r\n", $headers) . "\r\n\r\n" . $normalizedBody . "\r\n.";
        $expect($sendCommand($payload), [250], 'Envoi du corps refusé');
        $sendCommand('QUIT');
        fclose($socket);

        return true;
    } catch (Throwable $exception) {
        error_log('SMTP error: ' . $exception->getMessage());
        fclose($socket);
        return false;
    }
}

function encode_header(string $value): string
{
    return '=?UTF-8?B?' . base64_encode($value) . '?=';
}

function encode_address(string $name, string $email): string
{
    $safeName = $name ? encode_header($name) : '';
    return $safeName ? sprintf('%s <%s>', $safeName, $email) : $email;
}
