<?php

declare(strict_types=1);

require_once __DIR__ . '/database.php';
require_once __DIR__ . '/i18n.php';

function app_config(): array
{
    static $config = null;

    if ($config === null) {
        $config = require __DIR__ . '/../config.php';
    }

    return $config;
}

function ensure_session(): void
{
    if (session_status() !== PHP_SESSION_ACTIVE) {
        session_start();
    }
}

function theme_primary_color(): string
{
    return app_config()['theme']['primary_color'] ?? '#0c63e4';
}

function theme_logo_path(): string
{
    return app_config()['theme']['logo_path'] ?? 'assets/img/logo.svg';
}

function adjust_color(string $hexColor, float $ratio): string
{
    $hex = ltrim($hexColor, '#');
    if (strlen($hex) === 3) {
        $hex = $hex[0] . $hex[0] . $hex[1] . $hex[1] . $hex[2] . $hex[2];
    }

    if (!preg_match('/^[0-9a-fA-F]{6}$/', $hex)) {
        return '#0a53c1';
    }

    $ratio = max(-1, min(1, $ratio));
    $components = str_split($hex, 2);
    $result = '#';

    foreach ($components as $component) {
        $value = hexdec($component);
        if ($ratio < 0) {
            $value = (int) round($value * (1 + $ratio));
        } else {
            $value = (int) round($value + (255 - $value) * $ratio);
        }
        $result .= str_pad(dechex(max(0, min(255, $value))), 2, '0', STR_PAD_LEFT);
    }

    return $result;
}

function theme_hover_color(): string
{
    return adjust_color(theme_primary_color(), -0.1);
}

function theme_style_tag(): string
{
    $primary = htmlspecialchars(theme_primary_color(), ENT_QUOTES);
    $hover = htmlspecialchars(theme_hover_color(), ENT_QUOTES);

    return <<<HTML
<style>
:root {
  --primary-color: {$primary};
  --primary-color-hover: {$hover};
}
</style>
HTML;
}

function auth_secret_token(): string
{
    return app_config()['auth']['secret_token'];
}

function auth_session_key(): string
{
    return app_config()['auth']['session_key'];
}

function login_user(int $userId): void
{
    ensure_session();
    $_SESSION[auth_session_key()] = $userId;
}

function logout_user(): void
{
    ensure_session();
    unset($_SESSION[auth_session_key()]);
}

function current_user_id(): ?int
{
    ensure_session();
    return isset($_SESSION[auth_session_key()]) ? (int) $_SESSION[auth_session_key()] : null;
}

function require_auth(): void
{
    if (current_user_id() === null) {
        header('Location: login.php');
        exit;
    }
}

const FLASH_KEY = 'flash_messages';

function set_flash(string $status, string $message, array $errors = [], array $old = [], array $extras = []): void
{
    ensure_session();
    $_SESSION[FLASH_KEY] = [
        'status' => $status,
        'message' => $message,
        'errors' => $errors,
        'old' => $old,
        'extras' => $extras,
    ];
}

function consume_flash(): array
{
    ensure_session();

    $flash = $_SESSION[FLASH_KEY] ?? [
        'status' => null,
        'message' => null,
        'errors' => [],
        'old' => [],
        'extras' => [],
    ];

    unset($_SESSION[FLASH_KEY]);

    return $flash;
}

function csrf_token(): string
{
    ensure_session();

    if (empty($_SESSION['_csrf_token'])) {
        $_SESSION['_csrf_token'] = bin2hex(random_bytes(32));
    }

    return $_SESSION['_csrf_token'];
}

function csrf_field(): string
{
    return '<input type="hidden" name="_token" value="' . htmlspecialchars(csrf_token(), ENT_QUOTES) . '">';
}

function valid_csrf(?string $token): bool
{
    ensure_session();

    $sessionToken = $_SESSION['_csrf_token'] ?? '';

    return is_string($token) && $token !== '' && hash_equals($sessionToken, $token);
}

function client_identifier(): string
{
    return $_SERVER['REMOTE_ADDR'] ?? 'cli';
}

function rate_limit_key(string $name): string
{
    return $name . '_' . client_identifier();
}

function hit_rate_limit(string $key, int $maxAttempts, int $decaySeconds): bool
{
    ensure_session();
    $now = time();
    $bucket = $_SESSION['rate_limits'][$key] ?? ['attempts' => 0, 'expires' => $now + $decaySeconds];

    if ($now > $bucket['expires']) {
        $bucket = ['attempts' => 0, 'expires' => $now + $decaySeconds];
    }

    $bucket['attempts']++;
    $_SESSION['rate_limits'][$key] = $bucket;

    return $bucket['attempts'] > $maxAttempts;
}

function reset_rate_limit(string $key): void
{
    ensure_session();
    unset($_SESSION['rate_limits'][$key]);
}

function field_label(string $field, string $lang): string
{
    return translate('labels.' . $field, $lang);
}
